// Game_Music_Emu 0.4.0. http://www.slack.net/~ant/

#include "Nsf_Emu.h"

#include <string.h>
#include <stdio.h>

#include "blargg_endian.h"

#if !NSF_EMU_APU_ONLY
	#include "Nes_Namco_Apu.h"
	#include "Nes_Vrc6_Apu.h"
	#include "Nes_Fme7_Apu.h"
#endif

/* Copyright (C) 2003-2006 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
more details. You should have received a copy of the GNU Lesser General
Public License along with this module; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#include "blargg_source.h"

static Music_Emu* new_nsf_emu() { return BLARGG_NEW Nsf_Emu; }

gme_type_t_ const gme_nsf_type [1] = { &new_nsf_emu, &Nsf_Emu::read_info };

Nes_Emu::equalizer_t const Nes_Emu::nes_eq     = {  -1.0, 80 };
Nes_Emu::equalizer_t const Nes_Emu::famicom_eq = { -15.0, 80 };

int const vrc6_flag  = 0x01;
int const namco_flag = 0x10;
int const fme7_flag  = 0x20;

int Nsf_Emu::pcm_read( void* emu, nes_addr_t addr )
{
	return *((Nsf_Emu*) emu)->cpu::get_code( addr );
}

Nes_Emu::Nes_Emu( double gain_ )
{
	play_addr = 0;
	gain = gain_;
	apu.dmc_reader( pcm_read, this );
	vrc6 = NULL;
	namco = NULL;
	fme7 = NULL;
	Music_Emu::set_equalizer( nes_eq );
	
	memset( unmapped_code, Nes_Cpu::bad_opcode, sizeof unmapped_code );
	
	blargg_verify_byte_order();
}

Nes_Emu::~Nes_Emu()
{
	unload();
}

void Nsf_Emu::unload()
{
	#if !NSF_EMU_APU_ONLY
	{
		delete vrc6;
		vrc6 = NULL;
		
		delete namco;
		namco = NULL;
		
		delete fme7;
		fme7 = NULL;
	}
	#endif
	
	rom.clear();
	Music_Emu::unload();
}

const char** Nsf_Emu::voice_names() const
{
	static const char* base_names [] = {
		"Square 1", "Square 2", "Triangle", "Noise", "DMC"
	};
	static const char* namco_names [] = {
		"Square 1", "Square 2", "Triangle", "Noise", "DMC",
		"Namco 5&7", "Namco 4&6", "Namco 1-3"
	};
	static const char* vrc6_names [] = {
		"Square 1", "Square 2", "Triangle", "Noise", "DMC",
		"VRC6 Square 1", "VRC6 Square 2", "VRC6 Saw"
	};
	static const char* dual_names [] = {
		"Square 1", "Square 2", "Triangle", "Noise", "DMC",
		"VRC6.1,N106.5&7", "VRC6.2,N106.4&6", "VRC6.3,N106.1-3"
	};
	
	static const char* fme7_names [] = {
		"Square 1", "Square 2", "Triangle", "Noise", "DMC",
		"Square 3", "Square 4", "Square 5"
	};
	
	if ( namco )
		return vrc6 ? dual_names : namco_names;
	
	if ( vrc6 )
		return vrc6_names;
	
	if ( fme7 )
		return fme7_names;
	
	return base_names;
}

blargg_err_t Nsf_Emu::init_sound()
{
	if ( exp_flags & ~(namco_flag | vrc6_flag | fme7_flag) )
		return "NSF requires unsupported audio expansion hardware";
	
	set_voice_count( Nes_Apu::osc_count );
	
	double adjusted_gain = gain;
	
	#if NSF_EMU_APU_ONLY
	{
		if ( exp_flags )
			return "NSF requires expansion audio hardware";
	}
	#else
	{
		if ( exp_flags )
			set_voice_count( Nes_Apu::osc_count + 3 );
		
		if ( exp_flags & namco_flag )
		{
			namco = BLARGG_NEW Nes_Namco_Apu;
			CHECK_ALLOC( namco );
			adjusted_gain *= 0.75;
		}
		
		if ( exp_flags & vrc6_flag )
		{
			vrc6 = BLARGG_NEW Nes_Vrc6_Apu;
			CHECK_ALLOC( vrc6 );
			adjusted_gain *= 0.75;
		}
		
		if ( exp_flags & fme7_flag )
		{
			fme7 = BLARGG_NEW Nes_Fme7_Apu;
			CHECK_ALLOC( fme7 );
			adjusted_gain *= 0.75;
		}
		
		if ( namco ) namco->volume( adjusted_gain );
		if ( vrc6  ) vrc6 ->volume( adjusted_gain );
		if ( fme7  ) fme7 ->volume( adjusted_gain );
	}
	#endif
	
	apu.volume( adjusted_gain );
	
	return 0;
}

void Nsf_Emu::update_eq( blip_eq_t const& eq )
{
	apu.treble_eq( eq );
	
	#if !NSF_EMU_APU_ONLY
	{
		if ( namco ) namco->treble_eq( eq );
		if ( vrc6  ) vrc6 ->treble_eq( eq );
		if ( fme7  ) fme7 ->treble_eq( eq );
	}
	#endif
}

blargg_err_t Nsf_Emu::load( Data_Reader& in )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	return load( h, in );
}

blargg_err_t Nsf_Emu::load( header_t const& h, Data_Reader& in )
{
	header_ = h;
	unload();
	
	// check compatibility
	if ( 0 != memcmp( header_.tag, "NESM\x1A", 5 ) )
		return "Not an NSF file";
	if ( header_.vers != 1 )
		return "Unsupported NSF format";
	
	// sound and memory
	exp_flags = header_.chip_flags;
	blargg_err_t err = init_sound();
	if ( err )
		return err;
	
	// set up data
	nes_addr_t load_addr = get_le16( header_.load_addr );
	init_addr = get_le16( header_.init_addr );
	play_addr = get_le16( header_.play_addr );
	if ( !load_addr ) load_addr = rom_begin;
	if ( !init_addr ) init_addr = rom_begin;
	if ( !play_addr ) play_addr = rom_begin;
	if ( load_addr < rom_begin || init_addr < rom_begin )
		return "Invalid address in NSF";
	
	// set up rom
	total_banks = (in.remain() + load_addr % page_size + page_size - 1) / page_size;
	RETURN_ERR( rom.resize( total_banks * page_size ) );
	memset( &rom [0], 0, rom.size() );
	err = in.read( &rom [load_addr % page_size], in.remain() );
	if ( err )
	{
		unload();
		return err;
	}
	
	// bank switching
	int first_bank = (load_addr - rom_begin) / page_size;
	for ( int i = 0; i < bank_count; i++ )
	{
		unsigned bank = i - first_bank;
		initial_banks [i] = (bank < (unsigned) total_banks) ? bank : 0;
		
		if ( header_.banks [i] )
		{
			// bank-switched
			memcpy( initial_banks, header_.banks, sizeof initial_banks );
			break;
		}
	}
	
	// playback rate
	unsigned playback_rate = get_le16( header_.ntsc_speed );
	unsigned standard_rate = 0x411A;
	double clock_rate = 1789772.72727;
	play_period = 262 * 341L * 4 + 2;
	pal_only = false;
	
	if ( (header_.speed_flags & 3) == 1 )
	{
		pal_only = true;
		play_period = 33247 * master_clock_divisor;
		clock_rate = 1662607.125;
		standard_rate = 0x4E20;
		playback_rate = get_le16( header_.pal_speed );
	}
	
	if ( playback_rate && playback_rate != standard_rate )
		play_period = long (clock_rate * playback_rate * master_clock_divisor /
				1000000.0);
	
	// extra flags
	int extra_flags = header_.speed_flags;
	#if !NSF_EMU_EXTRA_FLAGS
		extra_flags = 0;
	#endif
	needs_long_frames = (extra_flags & 0x10) != 0;
	initial_pcm_dac = (extra_flags & 0x20) ? 0x3F : 0;
	
	set_track_count( header_.track_count );
	return setup_buffer( (long) (clock_rate + 0.5) );
}

void Nsf_Emu::set_voice( int i, Blip_Buffer* buf, Blip_Buffer*, Blip_Buffer* )
{
	if ( i < Nes_Apu::osc_count )
	{
		apu.osc_output( i, buf );
		return;
	}
	
	#if !NSF_EMU_APU_ONLY
	{
		if ( vrc6 ) vrc6->osc_output( i - Nes_Apu::osc_count, buf );
		
		if ( fme7 ) fme7->osc_output( i - Nes_Apu::osc_count, buf );
		
		if ( namco )
		{
			if ( i < 7 )
			{
				i &= 1;
				namco->osc_output( i + 4, buf );
				namco->osc_output( i + 6, buf );
			}
			else
			{
				for ( int n = 0; n < namco->osc_count / 2; n++ )
					namco->osc_output( n, buf );
			}
		}
	}
	#endif
}

void Nsf_Emu::cpu_jsr( nes_addr_t pc )
{
	cpu::r.pc = pc;
	cpu::push_byte( (badop_addr - 1) >> 8 );
	cpu::push_byte( (badop_addr - 1) );
}

void Nsf_Emu::start_track_( int track )
{
	require( rom.size() ); // file must be loaded
	
	Classic_Emu::start_track_( track );
	
	// reset memory
	cpu::reset( unmapped_code );
	cpu::map_code( 0x6000, sizeof sram, sram );
	memset( cpu::low_mem, 0, sizeof cpu::low_mem );
	memset( sram, 0, sizeof sram );
	
	// initial rom banks
	for ( int i = 0; i < bank_count; ++i )
		cpu::write( bank_select_addr + i, initial_banks [i] );
	
	// reset sound
	apu.reset( pal_only, initial_pcm_dac );
	apu.write_register( 0, 0x4015, 0x0F );
	apu.write_register( 0, 0x4017, needs_long_frames ? 0x80 : 0 );
	
	#if !NSF_EMU_APU_ONLY
	{
		if ( namco ) namco->reset();
		if ( vrc6  ) vrc6 ->reset();
		if ( fme7  ) fme7 ->reset();
	}
	#endif
	
	// reset cpu
	cpu::r.a        = track;
	cpu::r.x        = pal_only;
	cpu::r.y        = 0;
	cpu::r.sp       = 0xFF;
	cpu::r.status   = 0x04; // I flag set
	
	// first call
	play_ready = 4;
	play_extra = 0;
	next_play = play_period / master_clock_divisor;
	saved_state.pc = badop_addr;
	cpu_jsr( init_addr );
}

blip_time_t Nsf_Emu::run_clocks( blip_time_t duration, bool* )
{
	cpu::set_time( 0 );
	while ( cpu::time() < duration )
	{
		nes_time_t end = min( next_play, duration );
		end = min( end, cpu::time() + 32767 ); // allows CPU to use 16-bit time delta
		if ( cpu::run( end ) )
		{
			if ( cpu::r.pc != badop_addr )
			{
				log_error();
				cpu::r.pc++;
			}
			else
			{
				play_ready = 1;
				if ( saved_state.pc != badop_addr )
				{
					cpu::r = saved_state;
					saved_state.pc = badop_addr;
				}
				else
				{
					cpu::set_time( end );
				}
			}
		}
		
		if ( cpu::time() >= next_play )
		{
			nes_time_t period = (play_period + play_extra) / master_clock_divisor;
			play_extra = play_period - period * master_clock_divisor;
			next_play += period;
			if ( play_ready && !--play_ready )
			{
				check( saved_state.pc == badop_addr );
				if ( cpu::r.pc != badop_addr )
					saved_state = cpu::r;
				
				cpu_jsr( play_addr );
			}
		}
	}
	
	log_error( cpu::error_count() );
	cpu::clear_error_count();
	
	duration = cpu::time();
	next_play -= duration;
	check( next_play >= 0 );
	if ( next_play < 0 )
		next_play = 0;
	
	apu.end_frame( duration );
	
	#if !NSF_EMU_APU_ONLY
	{
		if ( namco ) namco->end_frame( duration );
		if ( vrc6  ) vrc6 ->end_frame( duration );
		if ( fme7  ) fme7 ->end_frame( duration );
	}
	#endif
	
	return duration;
}

void Nsf_Emu::cpu_write_misc( nes_addr_t addr, int data )
{
	#if !NSF_EMU_APU_ONLY
	{
		if ( namco )
		{
			switch ( addr )
			{
				case Nes_Namco_Apu::data_reg_addr:
					namco->write_data( cpu::time(), data );
					return;
				
				case Nes_Namco_Apu::addr_reg_addr:
					namco->write_addr( data );
					return;
			}
		}
		
		if ( addr >= Nes_Fme7_Apu::latch_addr && fme7 )
		{
			switch ( addr & Nes_Fme7_Apu::addr_mask )
			{
				case Nes_Fme7_Apu::latch_addr:
					fme7->write_latch( data );
					return;
				
				case Nes_Fme7_Apu::data_addr:
					fme7->write_data( cpu::time(), data );
					return;
			}
		}
		
		if ( vrc6 )
		{
			unsigned reg = addr & (Nes_Vrc6_Apu::addr_step - 1);
			unsigned osc = unsigned (addr - Nes_Vrc6_Apu::base_addr) / Nes_Vrc6_Apu::addr_step;
			if ( osc < Nes_Vrc6_Apu::osc_count && reg < Nes_Vrc6_Apu::reg_count )
			{
				vrc6->write_osc( cpu::time(), osc, reg, data );
				return;
			}
		}
	}
	#endif
	
	// unmapped write
	
	#ifndef NDEBUG
	{
		// some games write to $8000 and $8001 repeatedly
		if ( addr == 0x8000 || addr == 0x8001 ) return;
		
		// probably namco sound mistakenly turned on in mck
		if ( addr == 0x4800 || addr == 0xF800 ) return;
		
		// memory mapper?
		if ( addr == 0xFFF8 ) return;
		
		dprintf( "write_unmapped( 0x%04X, 0x%02X )\n", (unsigned) addr, (unsigned) data );
	}
	#endif
}

static void copy_nsf_fields( Nsf_Emu::header_t const& h, track_info_t* out )
{
	GME_COPY_FIELD( h, out, game );
	GME_COPY_FIELD( h, out, author );
	GME_COPY_FIELD( h, out, copyright );
	out->copy_field( out->system, (h.chip_flags ? "Famicom" : "Nintendo NES") );
}

blargg_err_t Nsf_Emu::track_info( track_info_t* out, int track ) const
{
	if ( Music_Emu::track_info( out, track ) ) { }
	copy_nsf_fields( header_, out );
	return out->error();
}

blargg_err_t Nsf_Emu::read_info( Data_Reader& in, track_info_t* out, int track )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	out->init();
	out->track_count = h.track_count;
	copy_nsf_fields( h, out );
	return out->error();
}
